// ----------------------------------------------------------------------------
// This reference design and source code is being provided on an "as-is" basis 
// and as an accommodation, and therefore all warranties, representations or 
// guarantees of any kind (whether express, implied or statutory) including, 
// without limitation, warranties of merchantability, non-infringement, or  
// fitness for a particular purpose, are specifically disclaimed.
//
// This source code may only be used in an Altera programmable logic device
// and may not be distributed without permission from Macnica Americas, Inc.  
// It is provided free of royalties or fees of any kind.
// ----------------------------------------------------------------------------
//
// Description  | A simple state machine for connecting register interface
//              | to avalon-mm.  Used to access the 3bit register
//              | space of Max10 dual-boot IP via I2C register space.
//              | 
//              | 
// Formatting   | Tabs set at 3 spaces.  Port names 'UPPER' case.
//              | Internal wires and registers are 'lower' case.
//              | Module names 'lower' case to match lower case filename.v
//              | Parameters are first character 'Upper' case.
//              | Active low signals are identified with '_n' or '_N'
//              | appended to the wire, register, or port name.
//              | 
//              | _r suffix on a register name indicates a simple
//              | re-registering of another wire or register.
//              | 
//              | _rr suffix on a register name indicates a second
//              | re-registering of a register.
//              | 
//              | _metastab indicates clock domain crossing
//
// ----------------------------------------------------------------------------


//--------------------------
//	Defines
//-----------
`define AVM_IDLE 3'b00
`define AVM_WRITE 3'b01
`define AVM_READ 3'b10


//--------------------------
//	Declaration and ports
//-----------
module avm_statemachine (

	//global
	input					IN_CLOCK,					//primary logic clock
	input					IN_RESET_N,					//synchronous reset
	
	//register interface
	input		[7:0]		IN_REG_ADDRCNTRL,			//address and control bits
	input		[15:0]	IN_REG_WDATA,				//write data bits
	output	[15:0]	OUT_REG_RDATA,				//read data bits

	//avalon interface
	output	[2:0]		OUT_AVM_ADDR,				//address
	output	[31:0]	OUT_AVM_WDATA,				//write data
	input		[31:0]	IN_AVM_RDATA,				//read data
	output				OUT_AVM_WRITE,				//write control
	output				OUT_AVM_READ				//read control
	
);


//--------------------------
//	Regs and Wires
//-----------
	reg					avm_write;
	reg					avm_read;
	reg	[31:0]		avm_rdata;
	reg	[2:0]			avm_state;
	wire					cntrl_w_n;
	wire					cntrl_start;

//--------------------------
//	StateMachine
//-----------
assign cntrl_w_n = IN_REG_ADDRCNTRL[4];
assign cntrl_start = IN_REG_ADDRCNTRL[5];
always @(posedge IN_CLOCK)
	begin
	if (!IN_RESET_N)
		begin
		avm_state <= `AVM_IDLE;
		avm_rdata <= 31'b0;
		end
	else
		begin
		case (avm_state[2:0])
		
		`AVM_IDLE:
			begin
			if (cntrl_start && (cntrl_w_n == 1'b1)) //start read
				begin
				avm_state <= `AVM_READ;
				end
			if (cntrl_start && (cntrl_w_n == 1'b0)) //start write
				begin
				avm_state <= `AVM_WRITE;
				end
			end
			
		`AVM_WRITE:
			begin
				avm_state <= `AVM_IDLE;
			end
			
		`AVM_READ:
			begin
				avm_state <= `AVM_IDLE;
				avm_rdata <= IN_AVM_RDATA;
			end
	
		endcase
		end
	end
	

//--------------------------
//	Assign outputs
//-----------
assign OUT_AVM_ADDR[2:0] = IN_REG_ADDRCNTRL[2:0];
assign OUT_AVM_READ = (avm_state[2:0] == `AVM_READ);
assign OUT_AVM_WRITE = (avm_state[2:0] == `AVM_WRITE);
assign OUT_AVM_WDATA[31:0]	= {16'b0,IN_REG_WDATA[15:0]};
assign OUT_REG_RDATA[15:0]	= avm_rdata[15:0];


endmodule

