// ----------------------------------------------------------------------------
// This reference design and source code is being provided on an "as-is" basis 
// and as an accommodation, and therefore all warranties, representations or 
// guarantees of any kind (whether express, implied or statutory) including, 
// without limitation, warranties of merchantability, non-infringement, or  
// fitness for a particular purpose, are specifically disclaimed.
//
// This source code may only be used in an Altera programmable logic device
// and may not be distributed without permission from Macnica Americas, Inc.  
// It is provided free of royalties or fees of any kind.
// ----------------------------------------------------------------------------
//
// Description  | Pattern Generator for pwm_control.v
//              | 
//              | Creates a back and forth sweeping motion acrosss
//              | the led array similar to that associated with the
//              | 80's nostallgia TV series "KnightRider"
//              | 
// Formatting   | Tabs set at 3 spaces.  Port names 'UPPER' case.
//              | Internal wires and registers are 'lower' case.
//              | Module names 'lower' case to match lower case filename.v
//              | Parameters are first character 'Upper' case.
//              | Active low signals are identified with '_n' or '_N'
//              | appended to the wire, register, or port name.
//              | 
//              | _r suffix on a register name indicates a simple
//              | re-registering of another wire or register.
//              | 
//              | _rr suffix on a register name indicates a second
//              | re-registering of a register.
//              | 
//              | _metastab indicates clock domain crossing
//
// ----------------------------------------------------------------------------

`timescale 1ns/1ps

//--------------------------
//	Defines
//-----------
	`define KNIGHT_LVL0 2'b00
	`define KNIGHT_LVL1 2'b01
	`define KNIGHT_LVL2 2'b10
	`define KNIGHT_LVL3 2'b11

//--------------------------
//	Declaration and ports
//-----------
module knightrider (
	
	//global
	input					IN_CLOCK,					//primary logic clock
	input					IN_RESET_N,					//synchronous reset
	
	//led
	output	[15:0]	OUT_LED_CNTRL				//binary encoded "pwm level" two bits per led
	
	);


//--------------------------
//	Parameters
//-----------
	parameter AdvanceCntMax = 250;
	parameter DelayCntMax = 10;


//--------------------------
//	Regs and Wires
//-----------
	reg		[15:0]	led_control;
	reg					up1down0;
	reg		[15:0]	advance_cnt;
	reg					advance_pulse;
	reg		[3:0]		state;
	reg		[7:0]		delay_cnt;
	reg					delay_reset;


//--------------------------
//	Logic
//-----------

//generate advance_pulse
//essentially a pre-scaler other logic
always @(posedge IN_CLOCK)
	begin
	if (!IN_RESET_N)
		begin
		advance_cnt <= 16'h0;
		advance_pulse <= 1'b0;
		end
	else
		begin
		if (advance_cnt == AdvanceCntMax)
			begin
			advance_cnt <= 16'h0;
			advance_pulse <= 1'b1;
			end
		else
			begin
			advance_cnt <=advance_cnt + 1'h1;
			advance_pulse <= 1'b0;
			end
		end
	end

//delay counter
//gives the "pause" between pattern runs
always @(posedge IN_CLOCK)
	begin
	if (!IN_RESET_N || delay_reset)
		begin
		delay_cnt <= 8'h0;
		end
	else
		begin
		if (advance_pulse)
			begin
			if (delay_cnt == DelayCntMax)
				begin
				delay_cnt <= delay_cnt;
				end
			else
				begin
				delay_cnt <= delay_cnt + 1'b1;
				end
			end
		end
	end

//Main State Machine
//Infinitely steps up thorugh 14 states, revereses down through same states, and repeats
always @(posedge IN_CLOCK)
	begin
	if (!IN_RESET_N)
		begin
		state <= 4'h0;
		up1down0 <= 1'b1;
		end
	else
		begin
		if (advance_pulse)
			begin
			case (state)
				(4'h0): //the "pause"
					begin
					delay_reset <= 1'b0;
					if (delay_cnt == DelayCntMax)
						begin
						state <= state + 1'b1;
						up1down0 <= 1'b1;
						end
					end
				(4'h1): 
					begin
					if (up1down0)
						begin
						state <= state + 1'b1;
						end
					else
						begin
						state <= state - 1'b1;
						delay_reset <= 1'b1;
						end
					end
				(4'h2):
					begin
					if (up1down0)
						begin
						state <= state + 1'b1;
						end
					else
						begin
						state <= state - 1'b1;
						end
					end			
				(4'h3):
					begin
					if (up1down0)
						begin
						state <= state + 1'b1;
						end
					else
						begin
						state <= state - 1'b1;
						end
					end				
				(4'h4):
					begin
					if (up1down0)
						begin
						state <= state + 1'b1;
						end
					else
						begin
						state <= state - 1'b1;
						end
					end
				(4'h5):
					begin
					if (up1down0)
						begin
						state <= state + 1'b1;
						end
					else
						begin
						state <= state - 1'b1;
						end
					end
				(4'h6):
					begin
					if (up1down0)
						begin
						state <= state + 1'b1;
						end
					else
						begin
						state <= state - 1'b1;
						end
					end
				(4'h7):
					begin
					if (up1down0)
						begin
						state <= state + 1'b1;
						end
					else
						begin
						state <= state - 1'b1;
						end
					end
				(4'h8):
					begin
					if (up1down0)
						begin
						state <= state + 1'b1;
						end
					else
						begin
						state <= state - 1'b1;
						end
					end
				(4'h9):
					begin
					if (up1down0)
						begin
						state <= state + 1'b1;
						end
					else
						begin
						state <= state - 1'b1;
						end
					end
				(4'ha):
					begin
					if (up1down0)
						begin
						state <= state + 1'b1;
						end
					else
						begin
						state <= state - 1'b1;
						end
					end
				(4'hb):
					begin
					if (up1down0)
						begin
						state <= state + 1'b1;
						end
					else
						begin
						state <= state - 1'b1;
						end
					end
				(4'hc):
					begin
					if (up1down0)
						begin
						state <= state + 1'b1;
						end
					else
						begin
						state <= state - 1'b1;
						end
					end					
				(4'hd):
					begin
					state <= state - 1'b1;
					up1down0 <= 1'b0;
					end	
			endcase //(state)		
			end
		end
	end

//create pwm "levels" from state creating the pattern
assign OUT_LED_CNTRL[15:0] = led_control[15:0];
always @(state)
	begin
	case (state)
		4'h0: led_control[15:0] = {`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0};
		4'h1: led_control[15:0] = {`KNIGHT_LVL1,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0};
		4'h2: led_control[15:0] = {`KNIGHT_LVL2,`KNIGHT_LVL1,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0};			
		4'h3: led_control[15:0] = {`KNIGHT_LVL3,`KNIGHT_LVL2,`KNIGHT_LVL1,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0};
		4'h4: led_control[15:0] = {`KNIGHT_LVL2,`KNIGHT_LVL3,`KNIGHT_LVL2,`KNIGHT_LVL1,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0};
		4'h5: led_control[15:0] = {`KNIGHT_LVL1,`KNIGHT_LVL2,`KNIGHT_LVL3,`KNIGHT_LVL2,`KNIGHT_LVL1,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0};	
		4'h6: led_control[15:0] = {`KNIGHT_LVL0,`KNIGHT_LVL1,`KNIGHT_LVL2,`KNIGHT_LVL3,`KNIGHT_LVL2,`KNIGHT_LVL1,`KNIGHT_LVL0,`KNIGHT_LVL0};
		4'h7: led_control[15:0] = {`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL1,`KNIGHT_LVL2,`KNIGHT_LVL3,`KNIGHT_LVL2,`KNIGHT_LVL1,`KNIGHT_LVL0};
		4'h8: led_control[15:0] = {`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL1,`KNIGHT_LVL2,`KNIGHT_LVL3,`KNIGHT_LVL2,`KNIGHT_LVL1};
		4'h9: led_control[15:0] = {`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL1,`KNIGHT_LVL2,`KNIGHT_LVL3,`KNIGHT_LVL2};
		4'ha: led_control[15:0] = {`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL1,`KNIGHT_LVL2,`KNIGHT_LVL3};
		4'hb: led_control[15:0] = {`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL1,`KNIGHT_LVL2};
		4'hc: led_control[15:0] = {`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL1};
		4'hd: led_control[15:0] = {`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0};
		default: led_control[15:0] = {`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0,`KNIGHT_LVL0};
	endcase
	end
	
endmodule