// ----------------------------------------------------------------------------
// This reference design and source code is being provided on an "as-is" basis 
// and as an accommodation, and therefore all warranties, representations or 
// guarantees of any kind (whether express, implied or statutory) including, 
// without limitation, warranties of merchantability, non-infringement, or  
// fitness for a particular purpose, are specifically disclaimed.
//
// This source code may only be used in an Altera programmable logic device
// and may not be distributed without permission from Macnica Americas, Inc.  
// It is provided free of royalties or fees of any kind.
// ----------------------------------------------------------------------------
//
// Description  | Implements user pushbutton control over pwms in dualboot_pwm design
//              | 
//              | Button0 cycles the mode (flash-->global-->pattern)
//              | Button1 cycles the level used in global mode (level0-->level1-->level2-->level3)
//              | Buttons are debounced and fire only once per press
//              | 
// Formatting   | Tabs set at 3 spaces.  Port names 'UPPER' case.
//              | Internal wires and registers are 'lower' case.
//              | Module names 'lower' case to match lower case filename.v
//              | Parameters are first character 'Upper' case.
//              | Active low signals are identified with '_n' or '_N'
//              | appended to the wire, register, or port name.
//              | 
//              | _r suffix on a register name indicates a simple
//              | re-registering of another wire or register.
//              | 
//              | _rr suffix on a register name indicates a second
//              | re-registering of a register.
//              | 
//              | _metastab indicates clock domain crossing
//
// ----------------------------------------------------------------------------


//--------------------------
//	Local Defines
//-----------

//global_cntrl states
`define PBUTT_LVL0 2'b00
`define PBUTT_LVL1 2'b01
`define PBUTT_LVL2 2'b10
`define PBUTT_LVL3 2'b11

//mode_cntrl states
`define PBUTT_GLOBAL_MODE 2'b00
`define PBUTT_FLASH_MODE 2'b10
`define PBUTT_PATTERN_MODE 2'b11

//--------------------------
//	Declaration and ports
//-----------
module pbutton (
	
	//global
	input					IN_CLOCK,
	input					IN_RESET_N,
	
	//buttons
	input		[1:0]		IN_BUTTON_N,
	
	//pwm control
	output	[1:0]		OUT_MODE_CNTRL,
	output	[1:0]		OUT_GLOBAL_CNTRL
	
);


//--------------------------
//	Regs and Wires
//-----------
	reg		[1:0]		mode_cntrl;
	reg		[1:0]		global_cntrl;
	reg		[1:0]		button_n;
	reg		[15:0]	debounce_cnt;
	reg		[1:0]		in_button_n_r;
	reg		[1:0]		in_button_n_rr;
	reg					button_event;
	
	
//--------------------------
//	Debounce Buttons
//-----------
always @(posedge IN_CLOCK)
	begin
	if (!IN_RESET_N)
		begin
		debounce_cnt <= 16'h00;
		in_button_n_r <= 2'b0;
		in_button_n_rr <= 2'b0;
		button_event <= 1'b0;
		end
	else
		 begin
		 in_button_n_r <= IN_BUTTON_N;
		 in_button_n_rr <= in_button_n_r;
		 if (in_button_n_r != in_button_n_rr) //reset debounce timeout on any edge
		 	begin
		 		debounce_cnt <= 16'h0000;
		 	end
		 if (debounce_cnt != 16'hffff) //increment timeout until saturation
		 	begin
		 		debounce_cnt <= debounce_cnt + 1'b1;
		 	end
		 if (debounce_cnt == 16'hfffe)  //capture and pulse event just before saturation
		 	begin
		 		button_n <= in_button_n_rr;
		 		button_event <= 1'b1;
		 	end
		 else
		 	begin
		 		button_event <= 1'b0;
		 	end
		 end
	end


//--------------------------
//	Button0 = cycle mode_cntrl
// Flash --> Pattern --> Global (repeat)
// (default to Flash)
//-----------
always @(posedge IN_CLOCK)
	begin
	if (!IN_RESET_N)
		begin
		mode_cntrl <= `PBUTT_FLASH_MODE;
		end
	else
		begin
		if (button_event && !button_n[0])
			begin
			case (mode_cntrl)
				(`PBUTT_FLASH_MODE):
					begin
					mode_cntrl <= `PBUTT_PATTERN_MODE;
					end
				(`PBUTT_PATTERN_MODE):
					begin
					mode_cntrl <= `PBUTT_GLOBAL_MODE;
					end
				(`PBUTT_GLOBAL_MODE):
					begin
					mode_cntrl <= `PBUTT_FLASH_MODE;
					end
			endcase
			end
		end
	end
	

//--------------------------
//	Button1 = cycle global_cntrl
// Level0 --> Level1 --> Level2 --> Level3 (repeat)
// (default to Level1)
//-----------
always @(posedge IN_CLOCK)
	begin
	if (!IN_RESET_N)
		begin
		global_cntrl <= `PBUTT_LVL1;
		end
	else
		begin
		if (button_event && !button_n[1])
			begin
			case (global_cntrl)
				(`PBUTT_LVL0):
					begin
					global_cntrl <= `PBUTT_LVL1;
					end
				(`PBUTT_LVL1):
					begin
					global_cntrl <= `PBUTT_LVL2;
					end
				(`PBUTT_LVL2):
					begin
					global_cntrl <= `PBUTT_LVL3;
					end
				(`PBUTT_LVL3):
					begin
					global_cntrl <= `PBUTT_LVL0;
					end					
			endcase
			end
		end
	end


//--------------------------
//	Assign outputs from state registers
//-----------
assign OUT_MODE_CNTRL[1:0] = mode_cntrl[1:0];
assign OUT_GLOBAL_CNTRL[1:0] = global_cntrl[1:0];


endmodule


