// ----------------------------------------------------------------------------
// This reference design and source code is being provided on an "as-is" basis 
// and as an accommodation, and therefore all warranties, representations or 
// guarantees of any kind (whether express, implied or statutory) including, 
// without limitation, warranties of merchantability, non-infringement, or  
// fitness for a particular purpose, are specifically disclaimed.
//
// This source code may only be used in an Altera programmable logic device
// and may not be distributed without permission from Macnica Americas, Inc.  
// It is provided free of royalties or fees of any kind.
// ----------------------------------------------------------------------------
//
// Description  | Controls one LED output in Macnica Odyssey IoT dualboot_pwm design
//              | 
//              | PWM "Value" accepts four levels: 00/01/10/11
//              | Three "Modes" 
//              | 	00 = Global = the Global input value controls pwm output
//              | 	01 = Channel = the Channel input value controls pwm output
//              | 	10 = Flash = no pwm.  output alternates off and on periods
//              |		11 = Pattern = the Pattern input value controls pwm output
//              | 
//              | Exact dutycycles and periods are dependent on supplied
//              | 	clock frequencies, comparison parameters, and are beyond the scope
//              | 	of this description.  Refer to the commented code.
//              | 
// Formatting   | Tabs set at 3 spaces.  Port names 'UPPER' case.
//              | Internal wires and registers are 'lower' case.
//              | Module names 'lower' case to match lower case filename.v
//              | Parameters are first character 'Upper' case.
//              | Active low signals are identified with '_n' or '_N'
//              | appended to the wire, register, or port name.
//              | 
//              | _r suffix on a register name indicates a simple
//              | re-registering of another wire or register.
//              | 
//              | _rr suffix on a register name indicates a second
//              | re-registering of a register.
//              | 
//              | _metastab indicates clock domain crossing
//
// ----------------------------------------------------------------------------

`timescale 1ns/1ps

//--------------------------
//	Declaration and ports
//-----------
module pwm_control (
	
	//global
	input					IN_CLOCK,					//primary logic clock
	input					IN_CLOCK_PWM,				//separate clock for counter (if needed)
	input					IN_CLOCK_FLASH,			//separate clock for flash (if needed)
	input					IN_RESET_N,					//synchronous reset
	
	//led
	output				OUT_PWM,						//pwm output
	
	//Control
	input		[1:0]		IN_MODE_CNTRL,				//controls mode (see header)	
	input		[1:0]		IN_GLOBAL_CNTRL,			//common PWM level for all channel
	input		[1:0]		IN_CHANNEL_CNTRL,			//separate PWM level for each channel
	input		[1:0]		IN_PATTERN_CNTRL			//separate PWM level for each channel

	);


//--------------------------
//	Regs and Wires
//-----------
	reg		[7:0]		pwm_cnt;
	reg		[7:0]		flash_cnt;
	reg		[7:0]		flash_cnt_r;
	reg		[7:0]		flash_cnt_rr;
	reg					pwm_output;


//--------------------------
//	Parameters
//-----------
	
	//in "PWM" mode: output high when cnt[7:0] > (threshold-1)
	//just trial and error for what gave good differentiation on LEDs
	parameter Level0Thresh = 256;		// "off"
	parameter Level1Thresh = 235;		//
	parameter Level2Thresh = 180;		//
	parameter Level3Thresh = 0;		// "on"
	
	//in "FLASH" mode: output toggles at FlashCntMax-1 when FlashEnable is 1
	parameter FlashCntMax = 150;
	parameter FlashEnable = 1;


//--------------------------
//	Logic
//-----------

	//The PWM counter:
	//Could easily share with FLASH counter.  Separated in this simple design.
	always @(posedge IN_CLOCK_PWM)
		begin
		if (!IN_RESET_N)
			begin
			pwm_cnt <= 8'b00;
			end
		else
			begin
			if (pwm_cnt == 8'hff)
				begin
				pwm_cnt <= 8'h00;
				end
			else
				begin
				pwm_cnt <= pwm_cnt + 1'b1;
				end
			end
		end

	//The FLASH counter:
	//Could easily share with PWM counter.  Separated in this simple design.
	always @(posedge IN_CLOCK_FLASH)
		begin
		if (!IN_RESET_N)
			begin
			flash_cnt <= 8'b00;
			end
		else
			begin
			if (flash_cnt == FlashCntMax)
				begin
				flash_cnt <= 8'h00;
				end
			else
				begin
				flash_cnt <= flash_cnt + 1'b1;
				end
			end
		end

	//Output control logic
	assign OUT_PWM = pwm_output;
	always @(posedge IN_CLOCK)
		begin
		if (!IN_RESET_N)
			begin
			pwm_output <= 1'b0;
			end
		else
			begin
			case (IN_MODE_CNTRL[1:0])

				(2'b00):	//GLOBAL MODE
					begin
					case (IN_GLOBAL_CNTRL[1:0])
						00: //level 0
							begin
							if (pwm_cnt > Level0Thresh)
								pwm_output <= 1'b1;
							else
								pwm_output <= 1'b0;
							end
						01: //level 1
							begin
							if (pwm_cnt > Level1Thresh)
								pwm_output <= 1'b1;
							else
								pwm_output <= 1'b0;
							end
						02: //level 2
							begin
							if (pwm_cnt > Level2Thresh)
								pwm_output <= 1'b1;
							else
								pwm_output <= 1'b0;
							end
						03: //level 3
							begin
							if (pwm_cnt > Level3Thresh)
								pwm_output <= 1'b1;
							else
								pwm_output <= 1'b0;
							end
					endcase //(IN_GLOBAL_CNTRL[1:0])
					end

				(2'b01): //CHANNEL MODE
					begin
					case (IN_CHANNEL_CNTRL[1:0])
						00: //level 0
							begin
							if (pwm_cnt > Level0Thresh)
								pwm_output <= 1'b1;
							else
								pwm_output <= 1'b0;
							end
						01: //level 1
							begin
							if (pwm_cnt > Level1Thresh)
								pwm_output <= 1'b1;
							else
								pwm_output <= 1'b0;
							end
						02: //level 2
							begin
							if (pwm_cnt > Level2Thresh)
								pwm_output <= 1'b1;
							else
								pwm_output <= 1'b0;
							end
						03: //level 3
							begin
							if (pwm_cnt > Level3Thresh)
								pwm_output <= 1'b1;
							else
								pwm_output <= 1'b0;
							end
					endcase //(IN_CHANNEL_CNTRL[1:0])
					end

				(2'b10): //FLASH MODE
					begin
					
					//sample flashcnt (since it runs at lower freq than this logic)
					//need sample to identify value change
					flash_cnt_r <= flash_cnt;
					flash_cnt_rr <= flash_cnt_r;
					
					if (FlashEnable == 1)
						begin
						if (flash_cnt_r != flash_cnt_rr)
							begin
							if ((pwm_output == 1'b1) && (flash_cnt == FlashCntMax))
								begin
								pwm_output <= 1'b0;
								end
							else if ((pwm_output == 1'b0) && (flash_cnt == FlashCntMax))
								begin
								pwm_output <= 1'b1;
								end
							end
						end
					else
						begin
						pwm_output <= 1'b0;
						end
					end

				(2'b11): //PATTERN MODE
					begin
					case (IN_PATTERN_CNTRL[1:0])
						00: //level 0
							begin
							if (pwm_cnt > Level0Thresh)
								pwm_output <= 1'b1;
							else
								pwm_output <= 1'b0;
							end
						01: //level 1
							begin
							if (pwm_cnt > Level1Thresh)
								pwm_output <= 1'b1;
							else
								pwm_output <= 1'b0;
							end
						02: //level 2
							begin
							if (pwm_cnt > Level2Thresh)
								pwm_output <= 1'b1;
							else
								pwm_output <= 1'b0;
							end
						03: //level 3
							begin
							if (pwm_cnt > Level3Thresh)
								pwm_output <= 1'b1;
							else
								pwm_output <= 1'b0;
							end
					endcase //(IN_PATTERN_CNTRL[1:0])
					end		

			endcase //(IN_MODE_CNTRL[1:0])
			end	
		end
	
endmodule