// ----------------------------------------------------------------------------
// This reference design and source code is being provided on an "as-is" basis 
// and as an accommodation, and therefore all warranties, representations or 
// guarantees of any kind (whether express, implied or statutory) including, 
// without limitation, warranties of merchantability, non-infringement, or  
// fitness for a particular purpose, are specifically disclaimed.
//
// This source code may only be used in an Altera programmable logic device
// and may not be distributed without permission from Macnica Americas, Inc.  
// It is provided free of royalties or fees of any kind.
// ----------------------------------------------------------------------------
//
// Description  | Demonstrate PWM control and dual configuration
//              | 
//              | I2C Register Map:
//              | Base = 0x30 (Wr base: 0x60, Rd base: 0x61)
//              |		00: AutoReg
//              | 		[0]	true when pwm control by I2C registers
//              | 		[1]	true when pwm control by buttons
//              | 		[2]	true when image0 loaded
//              | 		[3]	true when image1 loaded
//              | 		[5:4]	current mode(00:Global 01:Channel 10:Flash 11:Pattern)
//              | 		[7:6] current setting for Global mode (level of 0, 1, 2, 3)
//              | 	01: Global PWM Control
//              | 		[7:4]	unused
//              | 		[3:2]	mode (00:Global 01:Channel 10:Flash 11:Pattern)
//              | 		[1:0] global pwm level (Global pwm level of 0, 1, 2, 3)
//              | 	02: Channel PWM Control (upper)
//              | 		pwm for upper 4 channels.  level of 0, 1, 2, 3
//              | 		[7:6] pwm level for channel 7 
//              | 		[5:4] pwm level for channel 6 
//              | 		[3:2] pwm level for channel 5 
//              | 		[1:0] pwm level for channel 4 
//              | 	03: Channel PWM Control (lower)
//              | 		pwm for lower 4 channels.  level of 0, 1, 2, 3
//              | 		[7:6] pwm level for channel 3 
//              | 		[5:4] pwm level for channel 2 
//              | 		[3:2] pwm level for channel 1 
//              | 		[1:0] pwm level for channel 0 
//              | 	04: Avalon Indirect Control (access to dual configuration IP core)
//              | 		[7:6]	unused
//              | 		[5]	Start Indirect Access (self clears after setting)
//              | 		[4]	1=Indirect Read Access 0=Indirect Write Access
//              | 		[3]	unused
//              | 		[2:0]	Indirect Address
//              | 	05: Avalon Indirect WriteData (lower)
//              | 	06: Avalon Indirect WriteData (upper)
//              | 	09: Control Source
//              | 		[7:1]	unused
//              | 		[0]	control source (1=control from i2c register.  0=control from pushbuttons)
//              | 
//              | 
//              | Control by Buttons is default at start:
//              | 	Button0 cycles the mode (flash-->global-->pattern)
//              | 	Button1 cycles the level used in global mode (level0-->level1-->level2-->level3)
//              | 	Buttons are debounced and fire only once per press
//              | 
//              | 
//              | Mode is Flash at start and indicates which image is loaded:
//              | 	Outter LEDs flash == image0
//              | 	Inner LEDs flash == image1
//              | 
//              | 
//              | Platforms:
//              | macnica odyssey (target platform)
//              | 
//              | 
//              | 
// Formatting   | Tabs set at 3 spaces.  Port names 'UPPER' case.
//              | Internal wires and registers are 'lower' case.
//              | Module names 'lower' case to match lower case filename.v
//              | Parameters are first character 'Upper' case.
//              | Active low signals are identified with '_n' or '_N'
//              | appended to the wire, register, or port name.
//              | 
//              | _r suffix on a register name indicates a simple
//              | re-registering of another wire or register.
//              | 
//              | _rr suffix on a register name indicates a second
//              | re-registering of a register.
//              | 
//              | _metastab indicates clock domain crossing
//
// ----------------------------------------------------------------------------

`timescale 1ns/1ps

//--------------------------
//	Defines
//-----------
//
// There are also two different image<0/1>_topdefine.v files used to create two
// slightly different images for demonstration of dual image configuration.

//--------------------------
//	Declaration and ports
//-----------
module top (
	
	//global
	input					IN_CLOCK,
	
	//i2c
	input					IN_SCL,
	inout					INOUT_SDA,
	
	//gpio
	output	[7:0]		OUT_LED,
	input		[2:0]		IN_SWITCH,
	input		[1:0]		IN_BUTTON_N
);


//--------------------------
//	Regs and Wires
//-----------
	reg		[7:0]		autoreg;
	reg		[31:0]	reset_cnt;
	reg					reset_n;
	wire					locked;
	wire		[1:0]		i2c_mode_cntrl;
	wire		[1:0]		i2c_global_cntrl;
	wire					i2c_select;	
	wire		[1:0]		pbutt_mode_cntrl;
	wire		[1:0]		pbutt_global_cntrl;
	wire		[1:0]		ch7_cntrl;
	wire		[1:0]		ch6_cntrl;
	wire		[1:0]		ch5_cntrl;
	wire		[1:0]		ch4_cntrl;
	wire		[1:0]		ch3_cntrl;
	wire		[1:0]		ch2_cntrl;
	wire		[1:0]		ch1_cntrl;
	wire		[1:0]		ch0_cntrl;
	wire		[7:0]		unused;
	wire		[1:0]		ch7_pattern;
	wire		[1:0]		ch6_pattern;
	wire		[1:0]		ch5_pattern;
	wire		[1:0]		ch4_pattern;
	wire		[1:0]		ch3_pattern;
	wire		[1:0]		ch2_pattern;
	wire		[1:0]		ch1_pattern;
	wire		[1:0]		ch0_pattern;
	wire		[7:0]		out_led;
	wire					clk_core;
	wire					clk_pwm;
	wire					clk_flash;
	wire		[1:0]		mode_cntrl;
	wire		[1:0]		global_cntrl;
	wire		[7:0]		reg_addrcntrl;
	wire		[15:0]	reg_wdata;
	wire		[2:0]		avm_addr;
	wire		[31:0]	avm_wdata;
	wire		[31:0]	avm_rdata;
	wire					avm_write;
	wire					avm_read;


//--------------------------
//	PLL
//-----------
pll pll_inst (
	.inclk0 (IN_CLOCK),
	.c0 (clk_core),
	.c1 (clk_pwm),
	.c2 (clk_flash),
	.locked (locked)
	);


//--------------------------
//	Reset - Synchronous deassert after PLL locked
//-----------
always @(posedge IN_CLOCK)
	begin
	if (!locked)
		begin
		reset_cnt <= 32'h0000;
		reset_n <= 1'b0;
		end
	else
		begin
		if (reset_cnt == 32'hffff)
			begin
			reset_n <= 1'b1;
			end
		else
			begin
			reset_cnt <= reset_cnt + 1'b1;
			end
		end
	end


//--------------------------
//	The I2C Slave (Register Space)
//-----------
i2cSlave i2c_inst(
	.clk(clk_core),
	.rst(!reset_n),
	.sda(INOUT_SDA),
	.scl(IN_SCL),
	.reg0_auto_in(autoreg[7:0]),
	.reg1_global_out({unused[7:4],i2c_mode_cntrl[1:0],i2c_global_cntrl[1:0]}),
	.reg2_channel_upper_out({ch7_cntrl[1:0],ch6_cntrl[1:0],ch5_cntrl[1:0],ch4_cntrl[1:0]}),
	.reg3_channel_lower_out({ch3_cntrl[1:0],ch2_cntrl[1:0],ch1_cntrl[1:0],ch0_cntrl[1:0]}),
	.reg4_addrcntrl_out(reg_addrcntrl[7:0]),
	.reg5_wdata_lower_out(reg_wdata[7:0]),
	.reg6_wdata_upper_out(reg_wdata[15:8]),
	.reg9_control_source_out({unused[7:1],i2c_select})
);


//--------------------------
//	Push Button Control
//-----------
 pbutton pbutton_inst (
	//global
	.IN_CLOCK(clk_core),
	.IN_RESET_N(reset_n),
	//buttons	
	.IN_BUTTON_N(IN_BUTTON_N[1:0]),	
	//pwm control
	.OUT_MODE_CNTRL(pbutt_mode_cntrl[1:0]),
	.OUT_GLOBAL_CNTRL(pbutt_global_cntrl[1:0])
);


//--------------------------
//	Mux pwm control source
// selects between buttons and I2C
//-----------
assign mode_cntrl = (i2c_select ? i2c_mode_cntrl : pbutt_mode_cntrl);
assign global_cntrl = (i2c_select ? i2c_global_cntrl : pbutt_global_cntrl);


//--------------------------
//	KnightRider pattern generator
//-----------
knightrider pattern_inst (
	
	//global
	.IN_CLOCK(clk_flash),
	.IN_RESET_N(reset_n),
	
	//led
	.OUT_LED_CNTRL({
		ch7_pattern[1:0],
		ch6_pattern[1:0],
		ch5_pattern[1:0],
		ch4_pattern[1:0],
		ch3_pattern[1:0],
		ch2_pattern[1:0],
		ch1_pattern[1:0],
		ch0_pattern[1:0]
		})
	
	);


//--------------------------
//	Encode the auto register
// The auto reg is driven by HW and polled by software for indicator control
//-----------
always @(posedge clk_core)
	begin
	if (!reset_n)
		begin
		autoreg[7:0] <= 8'h00;
		end
	else
		begin		
		autoreg[0] <= i2c_select;
		autoreg[1] <= ~i2c_select;
		`ifdef TOP_IMAGE0
		autoreg[2] <= 1'b1; 
		`else
		autoreg[2] <= 1'b0;
		`endif
		`ifdef TOP_IMAGE1
		autoreg[3] <= 1'b1; 
		`else
		autoreg[3] <= 1'b0;
		`endif
		autoreg[5:4] <= mode_cntrl[1:0];
		autoreg[7:6] <= global_cntrl[1:0];
		end
	end


//--------------------------
//	PWM Control instances (8 total, one per LED) 
//-----------

// conditional compile used to identify image in dual configuration demonstration:
// first project compiles with TOP_IMAGE0 defined (enabling outer led flash only)
// second project compiles with TOP_IMAGE1 defined (enabling  inner led flash only)
// loading of image0/image1 is thus identified by outer/inner led flash
`ifdef TOP_IMAGE0
	defparam pwm0.FlashEnable = 1;
	defparam pwm1.FlashEnable = 1;
	defparam pwm2.FlashEnable = 0;
	defparam pwm3.FlashEnable = 0;
	defparam pwm4.FlashEnable = 0;
	defparam pwm5.FlashEnable = 0;
	defparam pwm6.FlashEnable = 1;
	defparam pwm7.FlashEnable = 1;
`elsif TOP_IMAGE1
	defparam pwm0.FlashEnable = 0;
	defparam pwm1.FlashEnable = 0;
	defparam pwm2.FlashEnable = 0;
	defparam pwm3.FlashEnable = 1;
	defparam pwm4.FlashEnable = 1;
	defparam pwm5.FlashEnable = 0;
	defparam pwm6.FlashEnable = 0;
	defparam pwm7.FlashEnable = 0;
`endif

assign OUT_LED[7:0] = ~out_led[7:0];
	
pwm_control pwm0 (	
	//global
	.IN_CLOCK(clk_core),
	.IN_CLOCK_PWM(clk_pwm),
	.IN_CLOCK_FLASH(clk_flash),
	.IN_RESET_N(reset_n),	
	//led
	.OUT_PWM(out_led[0]),
	//Control
	.IN_MODE_CNTRL(mode_cntrl[1:0]),
	.IN_GLOBAL_CNTRL(global_cntrl[1:0]),
	.IN_CHANNEL_CNTRL(ch0_cntrl[1:0]),
	.IN_PATTERN_CNTRL(ch0_pattern[1:0])
	);

pwm_control pwm1 (	
	//global
	.IN_CLOCK(clk_core),
	.IN_CLOCK_PWM(clk_pwm),
	.IN_CLOCK_FLASH(clk_flash),
	.IN_RESET_N(reset_n),	
	//led
	.OUT_PWM(out_led[1]),
	//Control
	.IN_MODE_CNTRL(mode_cntrl[1:0]),
	.IN_GLOBAL_CNTRL(global_cntrl[1:0]),
	.IN_CHANNEL_CNTRL(ch1_cntrl[1:0]),
	.IN_PATTERN_CNTRL(ch1_pattern[1:0])
	);
	
pwm_control pwm2 (	
	//global
	.IN_CLOCK(clk_core),
	.IN_CLOCK_PWM(clk_pwm),
	.IN_CLOCK_FLASH(clk_flash),
	.IN_RESET_N(reset_n),	
	//led
	.OUT_PWM(out_led[2]),	
	//Control
	.IN_MODE_CNTRL(mode_cntrl[1:0]),
	.IN_GLOBAL_CNTRL(global_cntrl[1:0]),
	.IN_CHANNEL_CNTRL(ch2_cntrl[1:0]),
	.IN_PATTERN_CNTRL(ch2_pattern[1:0])
	);
	
pwm_control pwm3 (	
	//global
	.IN_CLOCK(clk_core),
	.IN_CLOCK_PWM(clk_pwm),
	.IN_CLOCK_FLASH(clk_flash),
	.IN_RESET_N(reset_n),	
	//led
	.OUT_PWM(out_led[3]),
	//Control
	.IN_MODE_CNTRL(mode_cntrl[1:0]),
	.IN_GLOBAL_CNTRL(global_cntrl[1:0]),
	.IN_CHANNEL_CNTRL(ch3_cntrl[1:0]),
	.IN_PATTERN_CNTRL(ch3_pattern[1:0])
	);	

pwm_control pwm4 (	
	//global
	.IN_CLOCK(clk_core),
	.IN_CLOCK_PWM(clk_pwm),
	.IN_CLOCK_FLASH(clk_flash),
	.IN_RESET_N(reset_n),	
	//led
	.OUT_PWM(out_led[4]),	
	//Control
	.IN_MODE_CNTRL(mode_cntrl[1:0]),
	.IN_GLOBAL_CNTRL(global_cntrl[1:0]),
	.IN_CHANNEL_CNTRL(ch4_cntrl[1:0]),
	.IN_PATTERN_CNTRL(ch4_pattern[1:0])
	);

pwm_control pwm5 (	
	//global
	.IN_CLOCK(clk_core),
	.IN_CLOCK_PWM(clk_pwm),
	.IN_CLOCK_FLASH(clk_flash),
	.IN_RESET_N(reset_n),	
	//led
	.OUT_PWM(out_led[5]),	
	//Control
	.IN_MODE_CNTRL(mode_cntrl[1:0]),
	.IN_GLOBAL_CNTRL(global_cntrl[1:0]),
	.IN_CHANNEL_CNTRL(ch5_cntrl[1:0]),
	.IN_PATTERN_CNTRL(ch5_pattern[1:0])
	);
	
pwm_control pwm6 (	
	//global
	.IN_CLOCK(clk_core),
	.IN_CLOCK_PWM(clk_pwm),
	.IN_CLOCK_FLASH(clk_flash),
	.IN_RESET_N(reset_n),
	//led
	.OUT_PWM(out_led[6]),	
	//Control
	.IN_MODE_CNTRL(mode_cntrl[1:0]),
	.IN_GLOBAL_CNTRL(global_cntrl[1:0]),
	.IN_CHANNEL_CNTRL(ch6_cntrl[1:0]),
	.IN_PATTERN_CNTRL(ch6_pattern[1:0])
	);

pwm_control pwm7 (
	//global
	.IN_CLOCK(clk_core),
	.IN_CLOCK_PWM(clk_pwm),
	.IN_CLOCK_FLASH(clk_flash),
	.IN_RESET_N(reset_n),
	//led
	.OUT_PWM(out_led[7]),
	//Control
	.IN_MODE_CNTRL(mode_cntrl[1:0]),
	.IN_GLOBAL_CNTRL(global_cntrl[1:0]),
	.IN_CHANNEL_CNTRL(ch7_cntrl[1:0]),
	.IN_PATTERN_CNTRL(ch7_pattern[1:0])
	);


//--------------------------
//	Altera DualBoot Block
//-----------
dboot dboot_inst (
	.clk(clk_core),
	.nreset(reset_n),
	.avmm_rcv_address(avm_addr[2:0]),
	.avmm_rcv_read(avm_read),
	.avmm_rcv_writedata(avm_wdata[31:0]),
	.avmm_rcv_write(avm_write),
	.avmm_rcv_readdata(avm_rdata[31:0])
);


//--------------------------
//	Avalon StateMachine
//-----------
avm_statemachine avm_inst(
	.IN_CLOCK(clk_core),
	.IN_RESET_N(reset_n),
	//register interface
	.IN_REG_ADDRCNTRL(reg_addrcntrl[7:0]),
	.IN_REG_WDATA(reg_wdata[15:0]),
	.OUT_REG_RDATA(),
	//avalon interface
	.OUT_AVM_ADDR(avm_addr[2:0]),
	.OUT_AVM_WDATA(avm_wdata[31:0]),
	.IN_AVM_RDATA(avm_rdata[31:0]),
	.OUT_AVM_WRITE(avm_write),
	.OUT_AVM_READ(avm_read)
	);

	
endmodule